/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "e_cell.h"

#ifndef OPENSSL_NO_AES

#include "aes_core.c"
#include "aes_cbc.c"
#include "aes_cfb.c"
#include "aes_ofb.c"
#include "aes_ecb.c"

#include <openssl/evp.h>

ALIGNED(static AES_KEY g_key, SHARED_DATA_ALIGN);
ALIGNED(static unsigned char g_iv[EVP_MAX_IV_LENGTH], SHARED_DATA_ALIGN);
ALIGNED(static int g_num, SHARED_DATA_ALIGN);

static int spe_cipher_init(spe_cipher_request_t *cipher, const void *key, const void *iv)
	{
	unsigned int mode;

	ALIGN_HINT(cipher, SHARED_DATA_ALIGN, 0);

	mode = (cipher->flags & EVP_CIPH_MODE);

	if (cipher->encrypt || mode == EVP_CIPH_CFB_MODE || mode == EVP_CIPH_OFB_MODE)
		{
		AES_set_encrypt_key(key, cipher->key_size * 8, &g_key);
		}
	else
		{
		AES_set_decrypt_key(key, cipher->key_size * 8, &g_key);
		}

	ASSERT(cipher->iv_size <= sizeof(g_iv));

	memcpy(g_iv, iv, cipher->iv_size);

	g_num = 0;

	return 1;
	}

static int spe_cipher_do(spe_cipher_request_t *cipher, void *out, unsigned int *outl,
	const void *in, unsigned int inl)
	{
	unsigned int bs = cipher->block_size;
	unsigned int nid = cipher->nid;

	ALIGN_HINT(cipher, SHARED_DATA_ALIGN, 0);
	ALIGN_HINT(in, SHARED_DATA_ALIGN, 0);
	ALIGN_HINT(out, SHARED_DATA_ALIGN, 0);

	switch (nid)
		{
		case NID_aes_128_ecb:
		case NID_aes_192_ecb:
		case NID_aes_256_ecb:
		{
		unsigned int i = inl;
		while (i >= bs)
			{
			AES_ecb_encrypt(in, out, &g_key, cipher->encrypt);
			in += bs;
			out += bs;
			i -= bs;
			}
		*outl = inl - i;
		}
		break;
		case NID_aes_128_cbc:
		case NID_aes_192_cbc:
		case NID_aes_256_cbc:
			AES_cbc_encrypt(in, out, inl, &g_key, g_iv, cipher->encrypt);
			*outl = CEIL(inl, bs);
			break;
		case NID_aes_128_ofb128:
		case NID_aes_192_ofb128:
		case NID_aes_256_ofb128:
			AES_ofb128_encrypt(in, out, inl, &g_key, g_iv, &g_num);
			*outl = inl;
			break;
		case NID_aes_128_cfb128:
		case NID_aes_192_cfb128:
		case NID_aes_256_cfb128:
			AES_cfb128_encrypt(in, out, inl, &g_key, g_iv, &g_num, cipher->encrypt);
			*outl = inl;
			break;
		case NID_aes_128_cfb1:
		case NID_aes_192_cfb1:
		case NID_aes_256_cfb1:
			AES_cfb1_encrypt(in, out, inl * 8, &g_key, g_iv, &g_num, cipher->encrypt);
			*outl = inl;
			break;
		case NID_aes_128_cfb8:
		case NID_aes_192_cfb8:
		case NID_aes_256_cfb8:
			AES_cfb8_encrypt(in, out, inl, &g_key, g_iv, &g_num, cipher->encrypt);
			*outl = inl;
			break;
		default:
			DPRINTF("unknown NID (0x%08x)\n", nid);
			*outl = 0;
			break;
		}

	return 1;
	}

#include "spe_cipher.c"

#endif /* !OPENSSL_NO_AES */
